import tkinter as tk
import random
import sqlite3
import time
import math

# create the main window
window = tk.Tk()
window.title("Spin the Wheel")

# create the database connection
conn = sqlite3.connect('names.db')
c = conn.cursor()

# create the names table if it doesn't exist
c.execute('''CREATE TABLE IF NOT EXISTS names
             (name text)''')

# create the spin function
def spin(event=None):
    # get all the names from the database
    c.execute("SELECT name FROM names")
    result = c.fetchall()

    # if there are no names, display a message and return
    if not result:
        result_label.config(text="No names in the database.")
        return

    # pick a random name from the result list
    name = random.choice(result)[0]

    # calculate the angle at which the arrow should stop
    if result:
        angle = 360 / len(result)
    else:
        angle = 0

    stop_angle = result.index((name,)) * angle + angle / 2

    # calculate the angle for the selected name
    name_angle = (2 * math.pi / len(result)) * result.index((name,)) - math.pi / 2

    # rotate the arrow to the stop angle
    rotate(int(stop_angle))

    # rotate the arrow to the position of the selected name
    rotate(name_angle - stop_angle)

    # display the name in the result label
    result_label.config(text=name)



def spin_with_animation(event=None):
    # rotate the arrow by 360 degrees
    rotate(360)

    # call the spin function to select a random name
    spin()


# create the add function
def add(event=None):
    # get the name from the entry widget
    name = name_entry.get()

    # if the name is not empty, add it to the database
    if name:
        c.execute("INSERT INTO names VALUES (?)", (name,))
        conn.commit()
        # clear the entry widget
        name_entry.delete(0, tk.END)

        # fetch the updated list of names
        c.execute("SELECT name FROM names")
        updated_names = [row[0] for row in c.fetchall()]

        # redraw the canvas with the new names
        redraw_canvas(updated_names)

# create the remove function
def remove(event=None):
    # get the name from the entry widget
    name = name_entry.get()

    # if the name is not empty, remove it from the database
    if name:
        c.execute("DELETE FROM names WHERE name=?", (name,))
        conn.commit()
        # clear the entry widget
        name_entry.delete(0, tk.END)

        # fetch the updated list of names
        c.execute("SELECT name FROM names")
        updated_names = [row[0] for row in c.fetchall()]

        # redraw the canvas with the new names
        redraw_canvas(updated_names)

def redraw_canvas(updated_names):
    # clear the canvas
    wheel_canvas.delete("all")

    # calculate the start and end angles for each name
    angle = 360 / len(updated_names)
    start_angle = 90
    end_angle = start_angle + angle
    radius = 75

    # create a pie section for each name with a different background color
    for name in updated_names:
        # get a random background color for the pie section in the range of light to medium colors
        color = '#{0:02x}{1:02x}{2:02x}'.format(
            random.randint(180, 255),
            random.randint(180, 255),
            random.randint(180, 255),
        )

        # create the pie section with the background color
        wheel_canvas.create_arc(100, 100, 400, 400, start=start_angle, extent=angle, fill=color, outline='Black')

        # update the start and end angles for the next pie section
        start_angle += angle
        end_angle += angle

    angle = 2*math.pi/len(updated_names)
    start_angle = -math.pi/2
    for name in updated_names:
        x = 250 + radius * math.cos(start_angle + angle/2)
        y = 250 + radius * math.sin(start_angle + angle/2)
        text_id = wheel_canvas.create_text(x, y, text=name, font=('Arial', 14, 'bold'), angle=-math.degrees(start_angle + angle/2))
        start_angle += angle

    # re-draw the arrow
    global arrow
    arrow = wheel_canvas.create_polygon(240, 100, 260, 100, 250, 140, fill='black')

    # update the canvas
    wheel_canvas.update()

        

# create the rotate function
def rotate(angle):
    # calculate the center of the canvas
    center_x = 250
    center_y = 250
    
    # calculate the radius of the circle
    radius = 150
    
    # calculate the starting angle of the arrow
    start_angle = -math.pi/2
    
    # calculate the angle increment for each step
    step_angle = math.pi/180
    
    # loop through the specified angle range
    for i in range(angle):
        # calculate the current angle of the arrow
        arrow_angle = start_angle + i*step_angle
        
        # calculate the x and y coordinates of the arrow
        x = center_x + radius*math.cos(arrow_angle)
        y = center_y + radius*math.sin(arrow_angle)
        
        # calculate the x and y coordinates of the arrow pointer
        pointer_x = center_x + (radius - 40)*math.cos(arrow_angle)
        pointer_y = center_y + (radius - 40)*math.sin(arrow_angle)
        
        # calculate the coordinates of the arrow polygon
        arrow_coords = [
            x + 10*math.cos(arrow_angle + math.pi/2),
            y + 10*math.sin(arrow_angle + math.pi/2),
            pointer_x + 10*math.cos(arrow_angle),
            pointer_y + 10*math.sin(arrow_angle),
            x + 10*math.cos(arrow_angle - math.pi/2),
            y + 10*math.sin(arrow_angle - math.pi/2)
]

        
        # move the arrow to the new position
        wheel_canvas.coords(arrow, arrow_coords)
        
        # update the canvas
        wheel_canvas.update()
        
        # pause briefly to create the animation effect
        time.sleep(0.01)




# create the widgets
names = []
c.execute("SELECT name FROM names")
for row in c.fetchall():
    names.append(row[0])



wheel_canvas = tk.Canvas(window, width=500, height=500)
wheel_canvas.pack(side=tk.TOP)

# calculate the start and end angles for each name
if names:
    angle = 360 / len(names)
else:
    angle = 0

start_angle = 90
end_angle = start_angle + angle
radius = 75

# create a pie section for each name with a different background color
for name in names:
    # get a random background color for the pie section
    color = '#{0:02x}{1:02x}{2:02x}'.format(
            random.randint(180, 255),
            random.randint(180, 255),
            random.randint(180, 255),
        )

    # create the pie section with the background color
    wheel_canvas.create_arc(100, 100, 400, 400, start=start_angle, extent=angle, fill=color, outline='Black', width=1)

    # update the start and end angles for the next pie section
    start_angle += angle
    end_angle += angle

if names:
    angle = 2*math.pi/len(names)
    start_angle = -math.pi/2
    for name in names:
        x = 250 + radius * math.cos(start_angle + angle/2)
        y = 250 + radius * math.sin(start_angle + angle/2)
        text_id = wheel_canvas.create_text(x, y, text=name, font=('Arial', 14, 'bold'), angle=-math.degrees(start_angle + angle/2))
        start_angle += angle



arrow = wheel_canvas.create_polygon(240, 100, 260, 100, 250, 140, fill='black')
spin_button = tk.Button(window, text="Spin", command=lambda: [rotate(360), spin()])
spin_button.pack(side=tk.TOP)
spin_button.bind('<Return>', spin_with_animation)

name_entry = tk.Entry(window)
name_entry.pack(side=tk.LEFT)

add_button = tk.Button(window, text="Add", command=add)
add_button.pack(side=tk.LEFT)
add_button.bind('<Return>', add)

remove_button = tk.Button(window, text="Remove", command=remove)
remove_button.pack(side=tk.LEFT)
remove_button.bind('<Return>', remove)

result_label = tk.Label(window, text="", font=('Arial', 14, 'bold'))
result_label.pack(side=tk.BOTTOM)

# start the event loop
window.mainloop()

# close the database connection when the window is closed
conn.close()
